<?php
/* --------------------------------------------------------------
   PermissionDefinition.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Permission\Model;

use InvalidArgumentException;

/**
 * Class PermissionDefinition
 *
 * @package Gambio\Core\Auth\Model
 */
class PermissionDefinition implements \Gambio\Core\Permission\PermissionDefinition
{
    /**
     * @var string
     */
    private $identifier;
    
    /**
     * @var string
     */
    private $type;
    
    /**
     * @var string
     */
    private $action;
    
    /**
     * @var string[]
     */
    private $allowedTypes = ['page', 'controller', 'ajax_handler', 'route'];
    
    /**
     * @var string[]
     */
    private $allowedActions = ['read', 'write', 'delete'];
    
    
    /**
     * PermissionDefinition constructor.
     *
     * @param string $identifier
     * @param string $type
     * @param string $action
     */
    private function __construct(string $identifier, string $type, string $action)
    {
        if (!in_array(strtolower($type), $this->allowedTypes)) {
            throw new InvalidArgumentException('Type must be one of the following: ' . implode(', ',
                                                                                               $this->allowedTypes));
        }
        
        if (!in_array(strtolower($action), $this->allowedActions)) {
            throw new InvalidArgumentException('Action must be one of the following: ' . implode(', ',
                                                                                                 $this->allowedActions));
        }
        
        if (empty($identifier)) {
            throw new InvalidArgumentException('Identifier cannot be empty');
        }
        
        $this->identifier = $identifier;
        $this->type       = $type;
        $this->action     = $action;
    }
    
    
    /**
     * @param string $identifier
     * @param string $type
     * @param string $action
     *
     * @return PermissionDefinition
     */
    public static function create(string $identifier, string $type, string $action): PermissionDefinition
    {
        return new self ($identifier, $type, $action);
    }
    
    
    /**
     * @inheritDoc
     */
    public function identifier(): string
    {
        return $this->identifier;
    }
    
    
    /**
     * @inheritDoc
     */
    public function type(): string
    {
        return $this->type;
    }
    
    
    /**
     * @inheritDoc
     */
    public function action(): string
    {
        return $this->action;
    }
}